package com.seewo.open.sdk;

import com.seewo.open.sdk.auth.Account;
import com.seewo.open.sdk.constant.SeewoErrorCode;
import com.seewo.open.sdk.fastjson.JSON;
import com.seewo.open.sdk.fastjson.JSONObject;
import com.seewo.open.sdk.util.AntPathMatcher;
import com.seewo.open.sdk.util.SeewoStringUtils;
import com.seewo.open.sdk.util.SignUtils;

import java.io.IOException;
import java.net.ConnectException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Jiaze
 * @date 2018/10/12 下午4:29
 * @since 0.0.1
 */
public class DefaultSeewoClient implements SeewoClient {

    private Account account;

    private HttpClient httpClient;


    public DefaultSeewoClient(Account account) {
        this.account = account;
    }

    private static AntPathMatcher PATH_MATCHER = new AntPathMatcher();

    public <R extends OpenApiResult, P extends OpenApiParam> R invoke(OpenApiRequest<P, R> request) {
        setParam(request);
        HttpResponse response = execute(request);
        errorDetect(response);
        try {
            return request.getResponseClass().getConstructor(HttpResponse.class).newInstance(response);
        } catch (IllegalAccessException e) {
            throw new SeewoApiException(400, "构造函数" + request.getResponseClass() + "(com.seewo.open.sdk.HttpResponse) 不是共有的", null);
        } catch (NoSuchMethodException e) {
            throw new SeewoApiException(400, "没有找到共有构造函数" + request.getResponseClass() + "(com.seewo.open.sdk.HttpResponse)", null);
        } catch (Exception e) {
            throw new SeewoApiException(e);
        }
    }

    private void errorDetect(HttpResponse response) {
        String requestId = response.getHeaders().get("x-sw-req-id");
        if (response.getStatusCode() != 200) {
            String message = response.getHeaders().get("x-sw-message");
            if (SeewoStringUtils.isEmpty(message)) {
                message = response.getPhrase();
            }
            throw new SeewoApiException(response.getStatusCode(), message, requestId);
        }

        //如果业务出现异常，则抛出业务异常
        JSONObject jsonObject = JSON.parseObject(new String(response.getBodyBuffer(), StandardCharsets.UTF_8));
        if (jsonObject != null && jsonObject.getString("exception") != null && !jsonObject.getString("exception").equals("")) {
            String message = jsonObject.getString("message");
            if (message != null && !message.equals("")) {
                throw new BusinessException(message, requestId);
            } else {
                throw new BusinessException(jsonObject.getString("exception"), requestId);
            }
        }
    }

    /**
     * 设置网关参数，如 x-sw-sign 等
     *
     * @param request 请求对象
     */
    private <R extends OpenApiResult, P extends OpenApiParam> void setParam(OpenApiRequest<P, R> request) {
        String realPath = pattern2path(request.getPathVariables(), request.getPath());
        Map<String, String> extraHeaders = extraHeaders(request.getHeaders());
        request.addHeader("x-sw-app-id", account.getAppId());
        request.addHeader("x-sw-req-path", realPath);
        request.addHeader("x-sw-version", "3");
        request.addHeader("x-sw-timestamp", String.valueOf(System.currentTimeMillis()));
        request.addHeader("x-sw-sign-headers", SeewoStringUtils.setJoin(extraHeaders.keySet(), ","));
        request.addHeader("x-sw-sign-type", account.getSignType());
        if (!SeewoStringUtils.isEmptyEvenWhiteSpace(request.getPermissionId())) {
            request.addHeader("x-sw-data-permission", request.getPermissionId());
        }
        if (request.getBody().length != 0) {
            request.addHeader("x-sw-content-md5", md5(request.getBody()));
        }
        request.addHeader("x-sw-sign", sign(request));
        request.setPath(realPath);
    }

    private String md5(byte[] body) {
        return SignUtils.getContentMD5(body);
    }

    private Map<String, String> extraHeaders(Map<String, String> headers) {
        Map<String, String> extraHeaders = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            if (!entry.getKey().startsWith("x-sw-") && entry.getValue() != null) {
                extraHeaders.put(entry.getKey(), entry.getValue());
            }
        }
        return extraHeaders;
    }

    /**
     * 添加签名
     *
     * @param request 请求对象
     * @return 签名
     */
    private <R extends OpenApiResult, P extends OpenApiParam> String sign(OpenApiRequest<P, R> request) {
        Map<String, String> signParam = new HashMap<String, String>(request.getQueries());
        signParam.putAll(request.getHeaders());
        //去除参数x-sw-sign，即上一次签名的参数
        signParam.remove("x-sw-sign");
        String sign = null;
        try {
            sign = SignUtils.signRequest(signParam, account.getAppSecret(), account.getSignType());
        } catch (IOException e) {
            e.printStackTrace();
        }
        return sign;
    }

    public HttpResponse execute(HttpRequest request) {
        try {
            return getHttpClient().method(request);
        } catch (UnknownHostException e) {
            throw new SeewoApiException(SeewoErrorCode.UNKNOWN_HOST, e);
        } catch (ConnectException e) {
            throw new SeewoApiException(SeewoErrorCode.CONNECT_EXCEPTION, e);
        } catch (IOException e) {
            throw new SeewoApiException(SeewoErrorCode.IO_EXCEPTION, e);
        } catch (Exception e) {
            throw new SeewoApiException(e);
        }
    }

    private String pattern2path(Map<String, String> pathVars, String pattern) {
        if (pathVars.isEmpty()) return pattern;
        Map<String, String> varMap = PATH_MATCHER.extractUriTemplateVariables(pattern, pattern);
        String path = pattern;
        for (Map.Entry<String, String> entry : varMap.entrySet()) {
            path = path.replace(entry.getValue(), pathVars.get(entry.getKey()));
        }
        return path;
    }

    @Override
    public void setHttpClient(HttpClient httpClient) {
        this.httpClient = httpClient;
    }

    public HttpClient getHttpClient() {
        if (httpClient == null) httpClient = new DefaultHttpClient();
        return httpClient;
    }
}
