package com.seewo.open.sdk;


import com.seewo.open.sdk.fastjson.JSON;
import com.seewo.open.sdk.util.CodecUtils;

import java.lang.reflect.Field;
import java.util.*;

/**
 * @author Jiaze
 * @date 2017/10/10 9:25
 * @since 0.0.1
 */
public abstract class OpenApiParam {

    public static final String EMPTY_BODY = "";

    private Field[] declaredFields;

    private Map<String, String> pathVariables;

    private Map<String, String> headers;

    private Map<String, String> queries;

    private String body;

    /**
     * 子类可以重写此方法修改参与请求的字段。默认是全部字段
     *
     * @return 参与请求的字段
     */
    protected Field[] declaredFields() {
        if (declaredFields != null) {
            return declaredFields;
        } else {
            declaredFields = this.getClass().getDeclaredFields();
            return declaredFields;
        }
    }

    public Map<String, String> getPathVariables() {
        if(pathVariables == null)
        pathVariables = new HashMap<String, String>();
        for (Field field : declaredFields()) {
            ParameterPosition.PathVariable annotation;
            if ((annotation = field.getAnnotation(ParameterPosition.PathVariable.class)) != null) {
                String key = annotation.value();
                pathVariables.put(key, CodecUtils.urlEncode(String.valueOf(getValueFromField(field))));
            }
        }
        return pathVariables;
    }

    private synchronized void initParams() {
        pathVariables = new HashMap<String, String>();
        headers = new HashMap<String, String>();
        queries = new HashMap<String, String>();
        body = "";
        for (Field field : declaredFields()) {
            Object annotation;
            if ((annotation = field.getAnnotation(ParameterPosition.PathVariable.class)) != null) {
                String key = ((ParameterPosition.PathVariable) annotation).value();
                pathVariables.put(key, CodecUtils.urlEncode(String.valueOf(getValueFromField(field))));
            } else if ((annotation = field.getAnnotation(ParameterPosition.Query.class)) != null) {
                String key = ((ParameterPosition.Query) annotation).value();
                queries.put(key, String.valueOf(getValueFromField(field)));
            } else if ((annotation = field.getAnnotation(ParameterPosition.Header.class)) != null) {
                String key = ((ParameterPosition.Header) annotation).value();
                headers.put(key, String.valueOf(getValueFromField(field)));
            } else if (field.getAnnotation(ParameterPosition.RequestArray.class) != null) {
                if (Collection.class.isAssignableFrom(field.getType())) {
                    body = JSON.toJSONString(getValueFromField(field), true);
                }
            } else {
                body = JSON.toJSONString(getValueFromField(field), true);
            }
        }
    }

    private Object getValueFromField(Field field) {
        field.setAccessible(true);
        try {
            return field.get(this);
        } catch (IllegalAccessException e) {
            // impossible
            return null;
        }
    }

    //NotNull
    public Map<String, String> getQueries() {
        if(queries == null)
            queries = new HashMap<String, String>();
        for (Field field : declaredFields()) {
            ParameterPosition.Query annotation;
            if ((annotation = field.getAnnotation(ParameterPosition.Query.class)) != null) {
                String key = annotation.value();
                queries.put(key, String.valueOf(getValueFromField(field)));
            }
        }
        return queries;
    }

    public Map<String, String> getHeaders() {
        if(headers == null)
            headers = new HashMap<String, String>();
        for (Field field : declaredFields()) {
            ParameterPosition.Header annotation;
            if ((annotation = field.getAnnotation(ParameterPosition.Header.class)) != null) {
                String key = annotation.value();
                headers.put(key, String.valueOf(getValueFromField(field)));
            }
        }
        return headers;
    }

    public String getBody() {
        body = "";
        for (Field field : declaredFields()) {
            if ((field.getAnnotation(ParameterPosition.PathVariable.class)) != null
                    || (field.getAnnotation(ParameterPosition.Query.class)) != null
                    || (field.getAnnotation(ParameterPosition.Header.class)) != null) {
                //跳过头部等
                continue;
            }
            if (field.getAnnotation(ParameterPosition.RequestArray.class) != null) {
                if (Collection.class.isAssignableFrom(field.getType())) {
                    body = JSON.toJSONString(getValueFromField(field), true);
                }
            } else {
                body = JSON.toJSONString(getValueFromField(field), true);
            }
        }
        return body;
    }

    @Override
    public String toString() {
        return this.getClass() + "{" +
                "pathVariables=" + pathVariables +
                ", headers=" + headers +
                ", queries=" + queries +
                ", body='" + body + '\'' +
                '}';
    }
}
