package com.seewo.open.sdk;

import com.seewo.open.sdk.fastjson.JSON;
import com.seewo.open.sdk.util.CodecUtils;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

/**
 * @author Jiaze
 * @date 2017/10/10 9:25
 * @since 0.0.1
 */
public abstract class OpenApiResult {

    public OpenApiResult(HttpResponse httpResponse) {
        statusCode = httpResponse.getStatusCode();
        phrase = httpResponse.getPhrase();
        reqId = httpResponse.getHeaders().get("x-sw-req-id");
        message = httpResponse.getHeaders().get("x-sw-message");
        headers = httpResponse.getHeaders();
        body = CodecUtils.byte2string(httpResponse.getBodyBuffer());
        for (Field field : declaredFields()) {
            Object annotation;
            if ((annotation = field.getAnnotation(ParameterPosition.Header.class)) != null) {
                String key = ((ParameterPosition.Header) annotation).value();
                String value = headers.get(key);
                setValueToField(field, value);
            } else if ((annotation = field.getAnnotation(ParameterPosition.ResponseArray.class)) != null) {
                //bodyMap
                if (field.getClass().isAssignableFrom(Collection.class)) {
                    Class element = ((ParameterPosition.ResponseArray) annotation).element();
                    setValueToField(field, CodecUtils.parseJsonArray(body, element));
                }
            } else {
                //如果没有其他注解，认为是响应体
                setValueToField(field, JSON.parseObject(body, field.getType()));
            }
        }
    }

    @ParameterPosition.ResponseArray(element = String[].class)
    private Field[] declaredFields;

    private String reqId;

    private String message;

    private Integer statusCode;

    private String phrase;

    private Map<String, String> headers;

    private String body;

    /**
     * 子类可以重写此方法修改参与请求的字段。默认是全部字段
     *
     * @return 参与请求的字段
     */
    protected Field[] declaredFields() {
        if (declaredFields != null) {
            return declaredFields;
        } else {
            declaredFields = this.getClass().getDeclaredFields();
            return declaredFields;
        }
    }


    private void setValueToField(Field field, Object value) {
        field.setAccessible(true);
        try {
            field.set(this, value);
        } catch (IllegalAccessException e) {
            // impossible
        }
    }

    public String getReqId() {
        return reqId;
    }

    public void setReqId(String reqId) {
        this.reqId = reqId;
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers = headers;
    }

    public String getBody() {
        return body;
    }

    public void setBody(String body) {
        this.body = body;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    public void setStatusCode(Integer statusCode) {
        this.statusCode = statusCode;
    }

    public String getPhrase() {
        return phrase;
    }

    public void setPhrase(String phrase) {
        this.phrase = phrase;
    }

    @Override
    public String toString() {
        return this.getClass() + "{" +
                "reqId='" + reqId + '\'' +
                ", message='" + message + '\'' +
                ", statusCode=" + statusCode +
                ", phrase='" + phrase + '\'' +
                ", headers=" + headers + "\r\n" +
                ", body='" + body + '\'' +
                '}';
    }
}
