package com.seewo.open.sdk.util;


import com.seewo.open.sdk.Constants;
import com.seewo.open.sdk.SeewoApiException;
import com.seewo.open.sdk.constant.SeewoErrorCode;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.*;
import java.util.Map.Entry;


/**
 * 系统工具类。
 *
 * @author carver.gu
 * @since 1.0, Sep 12, 2009
 */
public abstract class SignUtils {
    private static final byte[] IV_BYTES = "0102030405060708".getBytes();
    private static final String AES = "AES";
    private static String intranetIp;
    private static final String MAC_HMAC_MD5 = "HmacMD5";

    private SignUtils() {
    }

    /**
     * 先对byte[]进行MD5再将二进制转成大写的十六进制
     */
    public static String getContentMD5(byte[] bytes) {
        if (bytes == null) {
            throw new IllegalArgumentException("bytes can not be null");
        }
        try {
            final MessageDigest md = MessageDigest.getInstance("MD5");
            md.reset();
            md.update(bytes);
            byte[] md5Result = md.digest();
            return byte2hex(md5Result);
        } catch (final NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("unknown algorithm MD5");
        }
    }

    /**
     * @param key    待加入参数的字段名
     * @param value  待加入参数的值
     * @param params 已存在的参数集合
     */
    public static void addParams(String key, String value, Map<String, String> params) {
        String existed = params.get(key);
        if (SeewoStringUtils.isEmptyEvenWhiteSpace(existed)) {
            params.put(key, value);
        } else {
            String sep = ",";
            String[] oldValues = existed.split(sep);
            String[] newValues = new String[oldValues.length + 1];
            System.arraycopy(oldValues, 0, newValues, 0, oldValues.length);
            newValues[oldValues.length] = value;
            Arrays.sort(newValues);
            StringBuilder sb = new StringBuilder();
            for (String target : newValues) {
                sb.append(target).append(sep);
            }
            params.put(key, sb.substring(0, sb.length() - 1));
        }
    }

    /**
     * 请求参数签名
     *
     * @param params     所有字符型的请求参数
     * @param secret     签名密钥
     * @param signMethod 签名方法，目前支持：md5, hmac_md5两种
     * @return 签名
     */
    public static String signRequest(Map<String, String> params, String secret, String signMethod)
            throws IOException {
        // 第一步：参数排序
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);

        // 第二步：把所有参数名和参数值串在一起
        StringBuilder query = new StringBuilder();
        if (Constants.SIGN_METHOD_MD5.equals(signMethod)) {
            query.append(secret);
        }
        for (String key : keys) {
            String value = params.get(key);
            if (SeewoStringUtils.areNotEmpty(key, value)) {
                query.append(key).append(value);
            }
        }

        // 第三步：使用MD5/HMAC加密
        byte[] bytes;
        if (Constants.SIGN_METHOD_HMAC.equals(signMethod)) {
            bytes = encryptHMAC(query.toString(), secret);
        } else {
            query.append(secret);
            bytes = encryptMD5(query.toString());
        }

        // 第四步：把二进制转化为大写的十六进制
        return byte2hex(bytes);
    }


    private static byte[] encryptHMAC(String data, String secret) throws IOException {
        byte[] bytes;
        try {
            SecretKey secretKey = new SecretKeySpec(secret.getBytes(Constants.CHARSET_UTF8), "HmacMD5");
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data.getBytes(Constants.CHARSET_UTF8));
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    /**
     * 对字符串采用UTF-8编码后，用MD5进行摘要。
     */
    private static byte[] encryptMD5(String data) throws IOException {
        return encryptMD5(data.getBytes(Constants.CHARSET_UTF8));
    }

    /**
     * 对字节流进行MD5摘要。
     */
    private static byte[] encryptMD5(byte[] data) throws IOException {
        byte[] bytes;
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            bytes = md.digest(data);
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    /**
     * 把字节流转换为十六进制表示方式。
     */
    private static String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();
        for (byte aByte : bytes) {
            String hex = Integer.toHexString(aByte & 0xFF);
            if (hex.length() == 1) {
                sign.append("0");
            }
            sign.append(hex.toUpperCase());
        }
        return sign.toString();
    }

    /**
     * 清除字典中值为空的项。
     *
     * @param <V> 泛型
     * @param map 待清除的字典
     * @return 清除后的字典
     */
    public static <V> Map<String, V> cleanupMap(Map<String, V> map) {
        if (map == null || map.isEmpty()) {
            return null;
        }

        Map<String, V> result = new HashMap<String, V>(map.size());
        Set<Entry<String, V>> entries = map.entrySet();

        for (Entry<String, V> entry : entries) {
            if (entry.getValue() != null) {
                result.put(entry.getKey(), entry.getValue());
            }
        }

        return result;
    }


    /**
     * 获取本机的局域网IP。
     */
    public static String getIntranetIp() {
        if (intranetIp == null) {
            try {
                intranetIp = InetAddress.getLocalHost().getHostAddress();
            } catch (Exception e) {
                intranetIp = "127.0.0.1";
            }
        }
        return intranetIp;
    }

    public static String aesEncrypt(String content, byte[] encryptKey) throws SeewoApiException {
        try {
            return aesEncrypt(content.getBytes(Constants.CHARSET_UTF8), encryptKey);
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * AES解密
     *
     * @param content    待解密的byte[]
     * @param decryptKey 解密密钥
     * @return 解密后的byte
     * @throws SeewoApiException
     */
    public static String aesDecrypt(String content, byte[] decryptKey) throws SeewoApiException {
        try {
            return new String(aesDecrypt(content.getBytes(Constants.CHARSET_UTF8), decryptKey), Constants.CHARSET_UTF8);
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * AES解密
     *
     * @param encryptBytes 待解密的byte[]
     * @param decryptKey   解密密钥
     * @return 解密后的byte
     * @throws SeewoApiException
     */
    public static byte[] aesDecrypt(byte[] encryptBytes, byte[] decryptKey) throws SeewoApiException {
        IvParameterSpec iv = new IvParameterSpec(IV_BYTES);
        Cipher cipher;
        try {
            cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(decryptKey, AES), iv);
            return cipher.doFinal(Base64.decode(encryptBytes));
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * AES加密
     *
     * @param content    待加密的内容
     * @param encryptKey 加密密钥
     * @return 加密后的byte[]
     * @throws SeewoApiException
     */
    public static String aesEncrypt(byte[] content, byte[] encryptKey) throws SeewoApiException {
        try {
            KeyGenerator kgen = KeyGenerator.getInstance(AES);
            kgen.init(new SecureRandom(encryptKey));

            IvParameterSpec iv = new IvParameterSpec(IV_BYTES);
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            cipher.init(Cipher.ENCRYPT_MODE, new SecretKeySpec(encryptKey, AES), iv);

            return base64Encode(cipher.doFinal(content));
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * . BASE64 编码(byte[]).
     *
     * @param src byte[] inputed string
     * @return String returned string
     */
    public static String base64Encode(byte[] src) throws SeewoApiException {
        try {
            return base64Encode(src, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * . BASE64 编码(byte[]).
     *
     * @param src         byte[] inputed string
     * @param charsetName
     * @return String returned string
     * @throws UnsupportedEncodingException
     */
    public static String base64Encode(byte[] src, String charsetName) throws UnsupportedEncodingException {
        byte[] res = Base64.encodeToByte(src, false);
        return (src != null ? new String(res, charsetName) : null);
    }

    /**
     * @param encryptText 被签名的字符串
     * @param encryptKey  密钥
     * @return
     * @throws Exception
     * @see #hmacMD5Encrypt(String, byte[])
     */
    public static String hmacMD5EncryptToBase64(String encryptText, byte[] encryptKey, int compressLen) throws SeewoApiException {
        try {
            return base64Encode(compress(hmacMD5Encrypt(encryptText, encryptKey), compressLen));
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * @param encryptText 被签名的字符串
     * @param encryptKey  密钥
     * @return
     * @throws Exception
     * @see #hmacMD5Encrypt(String, byte[])
     */
    public static String hmacMD5EncryptToBase64(String encryptText, byte[] encryptKey) throws SeewoApiException {
        try {
            return base64Encode(hmacMD5Encrypt(encryptText, encryptKey));
        } catch (Exception e) {
            throw new SeewoApiException(SeewoErrorCode.SEEWO_SIGN_ERROR, e);
        }
    }

    /**
     * 使用 HMAC-MD5 签名方法对对encryptText进行签名
     *
     * @param encryptText 被签名的字符串
     * @param encryptKey  密钥
     * @return
     * @throws Exception
     */
    public static byte[] hmacMD5Encrypt(String encryptText, byte[] encryptKey) throws Exception {
        // 根据给定的字节数组构造一个密钥,第二参数指定一个密钥算法的名称
        SecretKey secretKey = new SecretKeySpec(encryptKey, MAC_HMAC_MD5);
        // 生成一个指定 Mac 算法 的 Mac 对象
        Mac mac = Mac.getInstance(MAC_HMAC_MD5);
        // 用给定密钥初始化 Mac 对象
        mac.init(secretKey);

        byte[] text = encryptText.getBytes(Constants.CHARSET_UTF8);
        // 完成 Mac 操作
        return mac.doFinal(text);
    }

    /**
     * 生成滑动窗口
     *
     * @param input
     * @param slideSize
     * @return
     */
    public static List<String> getSlideWindows(String input, int slideSize) {
        List<String> windows = new ArrayList<String>();
        int startIndex = 0;
        int endIndex = 0;
        int currentWindowSize = 0;
        String currentWindow = null;

        while (endIndex < input.length() || currentWindowSize > slideSize) {
            boolean startsWithLetterOrDigit;
            if (currentWindow == null) {
                startsWithLetterOrDigit = false;
            } else {
                startsWithLetterOrDigit = isLetterOrDigit(currentWindow.charAt(0));
            }

            if (endIndex == input.length() && !startsWithLetterOrDigit) {
                break;
            }

            if (currentWindowSize == slideSize && !startsWithLetterOrDigit && isLetterOrDigit(input.charAt(endIndex))) {
                endIndex++;
                currentWindow = input.substring(startIndex, endIndex);
                currentWindowSize = 5;

            } else {
                if (endIndex != 0) {
                    if (startsWithLetterOrDigit) {
                        currentWindowSize -= 1;
                    } else {
                        currentWindowSize -= 2;
                    }
                    startIndex++;
                }

                while (currentWindowSize < slideSize && endIndex < input.length()) {
                    char currentChar = input.charAt(endIndex);
                    if (isLetterOrDigit(currentChar)) {
                        currentWindowSize += 1;
                    } else {
                        currentWindowSize += 2;
                    }
                    endIndex++;
                }
                currentWindow = input.substring(startIndex, endIndex);

            }
            windows.add(currentWindow);
        }
        return windows;
    }

    private static boolean isLetterOrDigit(char x) {
        if (0 <= x && x <= 127) {
            return true;
        }
        return false;
    }

    private static byte[] compress(byte[] input, int toLength) {
        if (toLength < 0) {
            return null;
        }
        byte[] output = new byte[toLength];
        for (int i = 0; i < output.length; i++) {
            output[i] = 0;
        }

        for (int i = 0; i < input.length; i++) {
            int index_output = i % toLength;
            output[index_output] ^= input[i];
        }

        return output;
    }

}
