package com.taobao.api.internal.tmc;

import com.taobao.api.internal.toplink.LinkException;
import com.taobao.api.internal.util.BooleanUtils;
import com.taobao.api.internal.util.NamedThreadFactory;
import com.taobao.api.internal.util.concurrent.RateLimiter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 拉模式TmcClient
 */
public class TmcPullClient extends TmcClient {
    private static final Log log = LogFactory.getLog(TmcPullClient.class);

    private static final String PULL_MODE = "pull_mode";

    /**
     * 默认流控值，设小一点比较安全
     * 来自tmc的第1条消息就会携带有最新的流控值，立刻就会把默认流控值给更新掉
     */
    private static final int DEFAULT_QPS_RATE = 32;

    /**
     * 流控器，控制消息处理速率
     */
    private final RateLimiter rateLimiter = RateLimiter.create(DEFAULT_QPS_RATE);

    /**
     * 当前流控值，当收到来自tmc的最新流控值时，会将最新流控值与该值做比较，如发现流控值变化了，就更新流控器&队列逻辑大小
     */
    private volatile int curQpsRate = DEFAULT_QPS_RATE;

    /**
     * 队列逻辑大小，会随当前的流控值改变，流控值越大，队列逻辑大小也越大，最大不会超过队列的真实大小
     * 计算公式 = 流控值 * 消息确认超时时间 * 80%(保险起见留20%的buffer，不能算的太死)
     * 队列逻辑大小 - 当前队列消息积压量 < 禁拉阈值时，不再主动拉取消息
     * 作用是避免队列积压消息太多导致消息确认超时TMC反复重发的问题
     */
    private volatile int queueLogicSize;

    /**
     * 禁拉阈值，调小使得队列允许的消息积压量上升，消息确认超时风险增加
     * 反之使得队列允许的消息积压量下降，消息确认超时风险减少
     * 日常测试时使用当前值能取得不错的效果
     */
    public static final int STOP_PULL_THRESHOLD = 32;

    /**
     * 消息确认超时时间，单位秒，该值为估计值
     * 实际值应为5秒（内存确认队列保存时间）+ 2 * 2 * 65536 / 1000秒（文件确认队列保存时间）+ 30秒（消息重投周期任务间隔）
     * 也就是约5分钟的样子
     * 估计值与实际值之间目前留了4分多钟的buffer，相当充足了
     */
    private static final int CONFIRM_TIMEOUT = 30;

    /**
     * 每200ms进行一次队列余量上报&消息主动拉取
     * guarded by this
     */
    private ScheduledExecutorService pullService;

    public TmcPullClient(String uri, String appKey, String appSecret, String groupName,
                         String minorGroup, String filterExp) {
        super(uri, appKey, appSecret, groupName, minorGroup, filterExp);
        // 初始化队列逻辑大小
        updateQueueLogicSize(curQpsRate);
    }

    @Override
    public void setQueueSize(int queueSize) {
        int minQueueSize = STOP_PULL_THRESHOLD + 2;
        if (queueSize < minQueueSize) {
            // 消息缓冲队列大小不能低于禁拉阈值，否则可能导致队列逻辑大小超过队列实际大小，起不到避免队列满的作用
            // 保险起见留一个2的buffer
            throw new IllegalArgumentException("queue size must greater than " + minQueueSize);
        }
        super.setQueueSize(queueSize);
        // 初始化队列逻辑大小
        updateQueueLogicSize(curQpsRate);
    }

    class InnerPullClient extends TmcClient.InnerClient {
        public InnerPullClient(TmcIdentity id) {
            super(id);
        }

        @Override
        protected Map<String, Object> createConnectHeaders() throws LinkException {
            Map<String, Object> connectHeaders = super.createConnectHeaders();
            // 建立连接时上报当前客户端是否是拉模式的
            // tmc侧连接处理逻辑见ChannelManager#handleConnect
            connectHeaders.put(PULL_MODE, BooleanUtils.convertToFlag(true));
            return connectHeaders;
        }

        @Override
        protected synchronized void onConnectSuccess() {
            super.onConnectSuccess();
            // 连接建立成功后启动pullService开始拉取消息
            // 这里要确保同时只有一个pullService在运行
            if(pullService != null) {
                pullService.shutdown();
            }
            pullService = new ScheduledThreadPoolExecutor(1,
                    new NamedThreadFactory("pull-service", true));
            pullService.scheduleAtFixedRate(new Runnable() {
                public void run() {
                    try {
                        int remainCapacity = getLogicRemainCapacity();
                        if(remainCapacity < STOP_PULL_THRESHOLD) {
                            // 队列当前还能容纳的消息量小于禁拉阈值，不提交拉请求
                            return;
                        }

                        // 主动去tmc拉一次消息，同时把队列当前还能容纳的消息量上报给tmc
                        // tmc侧处理逻辑见ProducerManager#sendPullEvent
                        Map<String, Object> pullRequest = new HashMap<String, Object>();
                        pullRequest.put(MessageFields.KIND, MessageKind.PullRequest);
                        pullRequest.put(MessageFields.REMAIN_CAPACITY, remainCapacity);
                        TmcClient.InnerClient innerClient = getClient();
                        if (innerClient != null && innerClient.isOnline()) {
                            innerClient.send(pullRequest);
                        }

                    } catch (Throwable e) {
                        log.error("pull error", e);
                    }
                }
            }, 200, 200, TimeUnit.MILLISECONDS);
        }
    }

    @Override
    protected InnerClient createInnerClient(TmcIdentity id) {
        return new InnerPullClient(id);
    }

    @Override
    protected void beforeSubmitMsgToQueue(Message message) {
        super.beforeSubmitMsgToQueue(message);
        Map<String, Object> rawMsg = message.getRaw();
        // 从消息中获取新的流控值并更新到本地
        Integer qpsRate = (Integer) rawMsg.get(MessageFields.DATA_QPS_RATE);
        if(qpsRate != null) {
            updateQpsRateIfChange(qpsRate);
        }
    }

    @Override
    protected void afterRetrieveMsgFromQueue() {
        super.afterRetrieveMsgFromQueue();
        // 流控
        rateLimiter.acquire();
    }

    @Override
    protected void pullRequest() {
        // 拉模式下每200ms就会主动拉一次tmc，不再需要推模式下每15s一次的兜底拉取逻辑了
    }

    @Override
    protected synchronized void onClose() {
        super.onClose();
        if(pullService != null) {
            pullService.shutdown();
        }
    }

    /**
     * clamp队列逻辑大小
     * 队列逻辑大小不能高于队列实际大小的80%，不能低于禁拉阈值，否则可能导致客户端队列满了也还在不断拉取消息or队列空了还不去主动拉取消息情况的出现
     * 保险起见两头各留一个1的buffer
     */
    private int clampQueueLogicSize(int newQueueLogicSize) {
        return Math.max(Math.min(newQueueLogicSize, ((int) Math.ceil(getQueueSize() * 0.8))-1), STOP_PULL_THRESHOLD+1);
    }

    /**
     * 根据流控值更新队列逻辑大小
     */
    private void updateQueueLogicSize(int qpsRate) {
        // 队列逻辑大小 = 流控值 * tmc确认超时时间 * 80%
        int newQueueLogicSize = (int) Math.ceil(qpsRate * CONFIRM_TIMEOUT * 0.8);
        this.queueLogicSize = clampQueueLogicSize(newQueueLogicSize);
    }

    /**
     * 检查流控值是否发生变化
     * 如发生变化更新本地RateLimiter&队列逻辑大小
     * 要上锁保证检查&更新操作的原子性
     * 因为本身这个方法被调的频率较低，目前是15秒左右1次，上锁并不会对性能造成太大影响
     * (tmc侧流控值未发生变化时，每15秒左右下发一次流控值，如发生变化，会立刻下发，流控值是人工调整的，不会频繁调整)
     * tmc侧流控值下发逻辑见NettyClient#notifyClientQpsRate
     */
    private synchronized void updateQpsRateIfChange(int qpsRate) {
        if(qpsRate > 0 && (curQpsRate != qpsRate)) {
            curQpsRate = qpsRate;
            rateLimiter.setRate(qpsRate);
            updateQueueLogicSize(qpsRate);
        }
    }

    /**
     * 根据队列逻辑大小计算当前队列还能容纳的消息数
     */
    public int getLogicRemainCapacity() {
        return queueLogicSize - getThreadPool().getQueue().size();
    }
}
